from flask import Flask, request, jsonify, session, send_from_directory
from flask_cors import CORS
import mysql.connector
from mysql.connector import pooling
import hashlib
import jwt
import datetime
from functools import wraps
import os
from dotenv import load_dotenv

load_dotenv()

app = Flask(__name__)
app.secret_key = os.getenv('SECRET_KEY', 'your-secret-key-change-this')
CORS(app, supports_credentials=True)

# Database Config
db_config = {
    'host': os.getenv('DB_HOST', 'localhost'),
    'user': os.getenv('DB_USER'),
    'password': os.getenv('DB_PASSWORD'),
    'database': os.getenv('DB_NAME'),
    'charset': 'utf8mb4',
    'collation': 'utf8mb4_unicode_ci',
    'pool_name': 'admin_pool',
    'pool_size': 5
}

connection_pool = pooling.MySQLConnectionPool(**db_config)

# ==========================================
# توابع کمکی
# ==========================================

def get_db():
    return connection_pool.get_connection()

def hash_password(password):
    return hashlib.sha256(password.encode()).hexdigest()

def generate_token(admin_id):
    payload = {
        'admin_id': admin_id,
        'exp': datetime.datetime.utcnow() + datetime.timedelta(days=7)
    }
    return jwt.encode(payload, app.secret_key, algorithm='HS256')

def verify_token(token):
    try:
        payload = jwt.decode(token, app.secret_key, algorithms=['HS256'])
        return payload['admin_id']
    except:
        return None

def token_required(f):
    @wraps(f)
    def decorated(*args, **kwargs):
        token = request.headers.get('Authorization')
        if not token:
            return jsonify({'error': 'توکن ارسال نشده'}), 401
        
        token = token.replace('Bearer ', '')
        admin_id = verify_token(token)
        
        if not admin_id:
            return jsonify({'error': 'توکن نامعتبر'}), 401
        
        conn = get_db()
        cursor = conn.cursor(dictionary=True)
        cursor.execute("SELECT * FROM admins WHERE id = %s AND is_active = TRUE", (admin_id,))
        admin = cursor.fetchone()
        cursor.close()
        conn.close()
        
        if not admin:
            return jsonify({'error': 'ادمین یافت نشد'}), 404
        
        return f(admin, *args, **kwargs)
    
    return decorated

# ==========================================
# Authentication
# ==========================================

@app.route('/api/auth/login', methods=['POST'])
def login():
    data = request.json
    username = data.get('username')
    password = data.get('password')
    
    if not username or not password:
        return jsonify({'error': 'اطلاعات ناقص'}), 400
    
    password_hash = hash_password(password)
    
    conn = get_db()
    cursor = conn.cursor(dictionary=True)
    cursor.execute(
        "SELECT * FROM admins WHERE username = %s AND password_hash = %s AND is_active = TRUE",
        (username, password_hash)
    )
    admin = cursor.fetchone()
    
    if not admin:
        cursor.close()
        conn.close()
        return jsonify({'error': 'نام کاربری یا رمز عبور اشتباه است'}), 401
    
    # به روزرسانی last_login
    cursor.execute("UPDATE admins SET last_login = NOW() WHERE id = %s", (admin['id'],))
    conn.commit()
    cursor.close()
    conn.close()
    
    token = generate_token(admin['id'])
    
    return jsonify({
        'token': token,
        'admin': {
            'id': admin['id'],
            'username': admin['username'],
            'full_name': admin['full_name'],
            'role': admin['role']
        }
    })

@app.route('/api/auth/me', methods=['GET'])
@token_required
def get_me(admin):
    return jsonify({'admin': admin})

# ==========================================
# Dashboard Stats
# ==========================================

@app.route('/api/dashboard/stats', methods=['GET'])
@token_required
def dashboard_stats(admin):
    conn = get_db()
    cursor = conn.cursor(dictionary=True)
    
    # تعداد کاربران
    cursor.execute("SELECT COUNT(*) as count FROM users")
    total_users = cursor.fetchone()['count']
    
    # تعداد کاربران امروز
    cursor.execute("SELECT COUNT(*) as count FROM users WHERE DATE(created_at) = CURDATE()")
    users_today = cursor.fetchone()['count']
    
    # درخواست های در انتظار
    cursor.execute("SELECT COUNT(*) as count FROM requests WHERE status = 'pending'")
    pending_requests = cursor.fetchone()['count']
    
    # درخواست های جدید
    cursor.execute("SELECT COUNT(*) as count FROM requests WHERE status = 'pending' AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)")
    new_requests = cursor.fetchone()['count']
    
    # درخواست های در حال انجام
    cursor.execute("SELECT COUNT(*) as count FROM requests WHERE status = 'in_progress'")
    in_progress_requests = cursor.fetchone()['count']
    
    # کل درخواست ها
    cursor.execute("SELECT COUNT(*) as count FROM requests")
    total_requests = cursor.fetchone()['count']
    
    # درخواست های تکمیل شده امروز
    cursor.execute("SELECT COUNT(*) as count FROM requests WHERE status = 'completed' AND DATE(completed_at) = CURDATE()")
    completed_today = cursor.fetchone()['count']
    
    # درآمد امروز
    cursor.execute("SELECT COALESCE(SUM(amount), 0) as total FROM transactions WHERE status = 'completed' AND DATE(completed_at) = CURDATE()")
    revenue_today = cursor.fetchone()['total']
    
    # درآمد ماه
    cursor.execute("SELECT COALESCE(SUM(amount), 0) as total FROM transactions WHERE status = 'completed' AND MONTH(completed_at) = MONTH(CURDATE())")
    revenue_month = cursor.fetchone()['total']
    
    cursor.close()
    conn.close()
    
    return jsonify({
        'total_users': total_users,
        'users_today': users_today,
        'pending_requests': pending_requests,
        'new_requests': new_requests,
        'in_progress_requests': in_progress_requests,
        'total_requests': total_requests,
        'completed_today': completed_today,
        'revenue_today': float(revenue_today),
        'revenue_month': float(revenue_month)
    })

# ==========================================
# Requests Management
# ==========================================

@app.route('/api/requests', methods=['GET'])
@token_required
def get_requests(admin):
    status = request.args.get('status', 'all')
    page = int(request.args.get('page', 1))
    per_page = int(request.args.get('per_page', 20))
    
    offset = (page - 1) * per_page
    
    conn = get_db()
    cursor = conn.cursor(dictionary=True)
    
    # ساخت کوئری
    where_clause = ""
    params = []
    
    if status != 'all':
        where_clause = "WHERE r.status = %s"
        params.append(status)
    
    # دریافت تعداد کل
    count_query = f"SELECT COUNT(*) as total FROM requests r {where_clause}"
    cursor.execute(count_query, params)
    total = cursor.fetchone()['total']
    
    # دریافت درخواست ها
    query = f"""
        SELECT r.*, u.first_name, u.last_name, u.username, 
               mp.title as prompt_title,
               CASE 
                   WHEN r.status = 'pending' AND r.created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR) THEN 'new'
                   ELSE r.status
               END as display_status
        FROM requests r
        LEFT JOIN users u ON r.user_id = u.id
        LEFT JOIN mega_prompts mp ON r.prompt_id = mp.id
        {where_clause}
        ORDER BY 
            CASE 
                WHEN r.status = 'pending' THEN 1 
                WHEN r.status = 'in_progress' THEN 2
                ELSE 3
            END,
            r.created_at ASC
        LIMIT %s OFFSET %s
    """
    
    params.extend([per_page, offset])
    cursor.execute(query, params)
    requests_data = cursor.fetchall()
    
    cursor.close()
    conn.close()
    
    return jsonify({
        'requests': requests_data,
        'total': total,
        'page': page,
        'per_page': per_page,
        'total_pages': (total + per_page - 1) // per_page
    })

@app.route('/api/requests/<int:request_id>', methods=['GET'])
@token_required
def get_request_detail(admin, request_id):
    conn = get_db()
    cursor = conn.cursor(dictionary=True)
    
    query = """
        SELECT r.*, 
               u.first_name, u.last_name, u.username, u.phone_number,
               u.age, u.gender, u.business_type,
               mp.title as prompt_title, mp.prompt_template,
               a.full_name as admin_name
        FROM requests r
        LEFT JOIN users u ON r.user_id = u.id
        LEFT JOIN mega_prompts mp ON r.prompt_id = mp.id
        LEFT JOIN admins a ON r.assigned_to_admin = a.id
        WHERE r.id = %s
    """
    
    cursor.execute(query, (request_id,))
    req = cursor.fetchone()
    
    if not req:
        cursor.close()
        conn.close()
        return jsonify({'error': 'درخواست یافت نشد'}), 404
    
    # دریافت اطلاعات شخصی سازی
    cursor.execute(
        "SELECT data_key, data_value FROM user_custom_data WHERE user_id = %s AND is_temporary = FALSE",
        (req['user_id'],)
    )
    custom_data = cursor.fetchall()
    
    # تغییر وضعیت به in_progress اگر pending بود
    if req['status'] == 'pending':
        cursor.execute(
            "UPDATE requests SET status = 'in_progress', started_at = NOW(), assigned_to_admin = %s WHERE id = %s",
            (admin['id'], request_id)
        )
        conn.commit()
        req['status'] = 'in_progress'
    
    cursor.close()
    conn.close()
    
    req['custom_data'] = {item['data_key']: item['data_value'] for item in custom_data}
    
    return jsonify({'request': req})

@app.route('/api/requests/<int:request_id>/response', methods=['POST'])
@token_required
def send_response(admin, request_id):
    data = request.json
    response_text = data.get('response_text')
    response_media_url = data.get('response_media_url')
    response_media_type = data.get('response_media_type', 'text')
    
    if not response_text and not response_media_url:
        return jsonify({'error': 'پاسخ خالی است'}), 400
    
    conn = get_db()
    cursor = conn.cursor(dictionary=True)
    
    # دریافت اطلاعات درخواست
    cursor.execute("SELECT * FROM requests WHERE id = %s", (request_id,))
    req = cursor.fetchone()
    
    if not req:
        cursor.close()
        conn.close()
        return jsonify({'error': 'درخواست یافت نشد'}), 404
    
    # به روزرسانی درخواست
    cursor.execute(
        """UPDATE requests SET 
           response_text = %s, 
           response_media_url = %s,
           response_media_type = %s,
           status = 'completed',
           completed_at = NOW()
           WHERE id = %s""",
        (response_text, response_media_url, response_media_type, request_id)
    )
    
    conn.commit()
    cursor.close()
    conn.close()
    
    # ارسال پاسخ به کاربر از طریق ربات
    # TODO: یکپارچه سازی با ربات تلگرام
    
    return jsonify({'success': True, 'message': 'پاسخ با موفقیت ارسال شد'})

# ==========================================
# Buttons Management
# ==========================================

@app.route('/api/buttons', methods=['GET'])
@token_required
def get_buttons(admin):
    conn = get_db()
    cursor = conn.cursor(dictionary=True)
    
    cursor.execute("""
        SELECT b.*, c.name as category_name,
               pb.button_text as parent_button_text
        FROM bot_buttons b
        LEFT JOIN categories c ON b.category_id = c.id
        LEFT JOIN bot_buttons pb ON b.parent_button_id = pb.id
        ORDER BY b.parent_button_id, b.order_index
    """)
    
    buttons = cursor.fetchall()
    cursor.close()
    conn.close()
    
    return jsonify({'buttons': buttons})

@app.route('/api/buttons', methods=['POST'])
@token_required
def create_button(admin):
    if not admin['can_manage_buttons']:
        return jsonify({'error': 'شما مجوز این کار را ندارید'}), 403
    
    data = request.json
    
    conn = get_db()
    cursor = conn.cursor()
    
    cursor.execute("""
        INSERT INTO bot_buttons 
        (category_id, parent_button_id, button_text, button_type, callback_data, 
         url, icon_emoji, is_active, is_under_construction, under_construction_message, order_index)
        VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s, %s, %s)
    """, (
        data.get('category_id'),
        data.get('parent_button_id'),
        data.get('button_text'),
        data.get('button_type'),
        data.get('callback_data'),
        data.get('url'),
        data.get('icon_emoji'),
        data.get('is_active', True),
        data.get('is_under_construction', False),
        data.get('under_construction_message'),
        data.get('order_index', 0)
    ))
    
    button_id = cursor.lastrowid
    conn.commit()
    cursor.close()
    conn.close()
    
    return jsonify({'success': True, 'button_id': button_id})

@app.route('/api/buttons/<int:button_id>', methods=['PUT'])
@token_required
def update_button(admin, button_id):
    if not admin['can_manage_buttons']:
        return jsonify({'error': 'شما مجوز این کار را ندارید'}), 403
    
    data = request.json
    
    conn = get_db()
    cursor = conn.cursor()
    
    cursor.execute("""
        UPDATE bot_buttons SET
        category_id = %s,
        parent_button_id = %s,
        button_text = %s,
        button_type = %s,
        callback_data = %s,
        url = %s,
        icon_emoji = %s,
        is_active = %s,
        is_under_construction = %s,
        under_construction_message = %s,
        order_index = %s
        WHERE id = %s
    """, (
        data.get('category_id'),
        data.get('parent_button_id'),
        data.get('button_text'),
        data.get('button_type'),
        data.get('callback_data'),
        data.get('url'),
        data.get('icon_emoji'),
        data.get('is_active'),
        data.get('is_under_construction'),
        data.get('under_construction_message'),
        data.get('order_index'),
        button_id
    ))
    
    conn.commit()
    cursor.close()
    conn.close()
    
    return jsonify({'success': True})

@app.route('/api/buttons/<int:button_id>', methods=['DELETE'])
@token_required
def delete_button(admin, button_id):
    if not admin['can_manage_buttons']:
        return jsonify({'error': 'شما مجوز این کار را ندارید'}), 403
    
    conn = get_db()
    cursor = conn.cursor()
    
    cursor.execute("DELETE FROM bot_buttons WHERE id = %s", (button_id,))
    
    conn.commit()
    cursor.close()
    conn.close()
    
    return jsonify({'success': True})

# ==========================================
# Prompts Management
# ==========================================

@app.route('/api/prompts', methods=['GET'])
@token_required
def get_prompts(admin):
    conn = get_db()
    cursor = conn.cursor(dictionary=True)
    
    cursor.execute("""
        SELECT mp.*, c.name as category_name, b.button_text
        FROM mega_prompts mp
        LEFT JOIN categories c ON mp.category_id = c.id
        LEFT JOIN bot_buttons b ON mp.button_id = b.id
        ORDER BY mp.created_at DESC
    """)
    
    prompts = cursor.fetchall()
    cursor.close()
    conn.close()
    
    return jsonify({'prompts': prompts})

@app.route('/api/prompts', methods=['POST'])
@token_required
def create_prompt(admin):
    if not admin['can_manage_prompts']:
        return jsonify({'error': 'شما مجوز این کار را ندارید'}), 403
    
    data = request.json
    
    conn = get_db()
    cursor = conn.cursor()
    
    # ساخت slug
    import re
    slug = re.sub(r'[^a-zA-Z0-9]+', '-', data.get('title')).lower()
    
    cursor.execute("""
        INSERT INTO mega_prompts
        (category_id, button_id, title, slug, description, prompt_template, 
         output_type, is_active, is_premium)
        VALUES (%s, %s, %s, %s, %s, %s, %s, %s, %s)
    """, (
        data.get('category_id'),
        data.get('button_id'),
        data.get('title'),
        slug,
        data.get('description'),
        data.get('prompt_template'),
        data.get('output_type'),
        data.get('is_active', True),
        data.get('is_premium', False)
    ))
    
    prompt_id = cursor.lastrowid
    conn.commit()
    cursor.close()
    conn.close()
    
    return jsonify({'success': True, 'prompt_id': prompt_id})

# ==========================================
# Bot Texts Management
# ==========================================

@app.route('/api/bot-texts', methods=['GET'])
@token_required
def get_bot_texts(admin):
    conn = get_db()
    cursor = conn.cursor(dictionary=True)
    
    cursor.execute("SELECT * FROM bot_texts ORDER BY text_key")
    texts = cursor.fetchall()
    
    cursor.close()
    conn.close()
    
    return jsonify({'texts': texts})

@app.route('/api/bot-texts/<int:text_id>', methods=['PUT'])
@token_required
def update_bot_text(admin, text_id):
    if not admin['can_manage_settings']:
        return jsonify({'error': 'شما مجوز این کار را ندارید'}), 403
    
    data = request.json
    
    conn = get_db()
    cursor = conn.cursor()
    
    cursor.execute("""
        UPDATE bot_texts SET text_value = %s WHERE id = %s
    """, (data.get('text_value'), text_id))
    
    conn.commit()
    cursor.close()
    conn.close()
    
    return jsonify({'success': True})

# ==========================================
# Settings Management
# ==========================================

@app.route('/api/settings', methods=['GET'])
@token_required
def get_settings(admin):
    conn = get_db()
    cursor = conn.cursor(dictionary=True)
    
    cursor.execute("SELECT * FROM system_settings ORDER BY setting_key")
    settings = cursor.fetchall()
    
    cursor.close()
    conn.close()
    
    return jsonify({'settings': settings})

@app.route('/api/settings/<int:setting_id>', methods=['PUT'])
@token_required
def update_setting(admin, setting_id):
    if not admin['can_manage_settings']:
        return jsonify({'error': 'شما مجوز این کار را ندارید'}), 403
    
    data = request.json
    
    conn = get_db()
    cursor = conn.cursor()
    
    cursor.execute("""
        UPDATE system_settings SET setting_value = %s WHERE id = %s
    """, (data.get('setting_value'), setting_id))
    
    conn.commit()
    cursor.close()
    conn.close()
    
    return jsonify({'success': True})

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=5000, debug=True)