from flask import Flask, request, jsonify
from flask_cors import CORS
import mysql.connector
from mysql.connector import pooling
import hashlib
import jwt
import datetime
from functools import wraps
import os

app = Flask(__name__)
app.secret_key = os.getenv('SECRET_KEY', 'change-this-secret-key')
CORS(app, supports_credentials=True)

db_config = {
    'host': os.getenv('DB_HOST', 'localhost'),
    'user': os.getenv('DB_USER'),
    'password': os.getenv('DB_PASSWORD'),
    'database': os.getenv('DB_NAME'),
    'charset': 'utf8mb4',
    'collation': 'utf8mb4_unicode_ci',
    'pool_name': 'admin_pool',
    'pool_size': 5
}

connection_pool = pooling.MySQLConnectionPool(**db_config)

def get_db():
    return connection_pool.get_connection()

def hash_password(password):
    return hashlib.sha256(password.encode()).hexdigest()

def generate_token(admin_id):
    payload = {
        'admin_id': admin_id,
        'exp': datetime.datetime.utcnow() + datetime.timedelta(days=7)
    }
    return jwt.encode(payload, app.secret_key, algorithm='HS256')

def verify_token(token):
    try:
        payload = jwt.decode(token, app.secret_key, algorithms=['HS256'])
        return payload['admin_id']
    except:
        return None

def token_required(f):
    @wraps(f)
    def decorated(*args, **kwargs):
        token = request.headers.get('Authorization')
        if not token:
            return jsonify({'error': 'توکن ارسال نشده'}), 401
        
        token = token.replace('Bearer ', '')
        admin_id = verify_token(token)
        
        if not admin_id:
            return jsonify({'error': 'توکن نامعتبر'}), 401
        
        conn = get_db()
        cursor = conn.cursor(dictionary=True)
        cursor.execute("SELECT * FROM admins WHERE id = %s AND is_active = TRUE", (admin_id,))
        admin = cursor.fetchone()
        cursor.close()
        conn.close()
        
        if not admin:
            return jsonify({'error': 'ادمین یافت نشد'}), 404
        
        return f(admin, *args, **kwargs)
    return decorated

@app.route('/api/auth/login', methods=['POST'])
def login():
    data = request.json
    username = data.get('username')
    password = data.get('password')
    
    if not username or not password:
        return jsonify({'error': 'اطلاعات ناقص'}), 400
    
    password_hash = hash_password(password)
    
    conn = get_db()
    cursor = conn.cursor(dictionary=True)
    cursor.execute(
        "SELECT * FROM admins WHERE username = %s AND password_hash = %s AND is_active = TRUE",
        (username, password_hash)
    )
    admin = cursor.fetchone()
    
    if not admin:
        cursor.close()
        conn.close()
        return jsonify({'error': 'نام کاربری یا رمز عبور اشتباه است'}), 401
    
    cursor.execute("UPDATE admins SET last_login = NOW() WHERE id = %s", (admin['id'],))
    conn.commit()
    cursor.close()
    conn.close()
    
    token = generate_token(admin['id'])
    
    return jsonify({
        'token': token,
        'admin': {
            'id': admin['id'],
            'username': admin['username'],
            'full_name': admin['full_name'],
            'role': admin['role']
        }
    })

@app.route('/api/dashboard/stats', methods=['GET'])
@token_required
def dashboard_stats(admin):
    conn = get_db()
    cursor = conn.cursor(dictionary=True)
    
    cursor.execute("SELECT COUNT(*) as count FROM users")
    total_users = cursor.fetchone()['count']
    
    cursor.execute("SELECT COUNT(*) as count FROM requests WHERE status = 'pending'")
    pending_requests = cursor.fetchone()['count']
    
    cursor.execute("SELECT COUNT(*) as count FROM requests")
    total_requests = cursor.fetchone()['count']
    
    cursor.close()
    conn.close()
    
    return jsonify({
        'total_users': total_users,
        'pending_requests': pending_requests,
        'total_requests': total_requests
    })

@app.route('/api/requests', methods=['GET'])
@token_required
def get_requests(admin):
    status = request.args.get('status', 'all')
    page = int(request.args.get('page', 1))
    per_page = int(request.args.get('per_page', 20))
    
    offset = (page - 1) * per_page
    
    conn = get_db()
    cursor = conn.cursor(dictionary=True)
    
    where_clause = ""
    params = []
    
    if status != 'all':
        where_clause = "WHERE r.status = %s"
        params.append(status)
    
    count_query = f"SELECT COUNT(*) as total FROM requests r {where_clause}"
    cursor.execute(count_query, params)
    total = cursor.fetchone()['total']
    
    query = f"""
        SELECT r.*, u.first_name, u.last_name, u.username
        FROM requests r
        LEFT JOIN users u ON r.user_id = u.id
        {where_clause}
        ORDER BY r.created_at DESC
        LIMIT %s OFFSET %s
    """
    
    params.extend([per_page, offset])
    cursor.execute(query, params)
    requests_data = cursor.fetchall()
    
    cursor.close()
    conn.close()
    
    return jsonify({
        'requests': requests_data,
        'total': total,
        'page': page,
        'per_page': per_page,
        'total_pages': (total + per_page - 1) // per_page
    })

@app.route('/api/requests/<int:request_id>', methods=['GET'])
@token_required
def get_request_detail(admin, request_id):
    conn = get_db()
    cursor = conn.cursor(dictionary=True)
    
    query = """
        SELECT r.*, u.first_name, u.last_name, u.username, u.phone_number
        FROM requests r
        LEFT JOIN users u ON r.user_id = u.id
        WHERE r.id = %s
    """
    
    cursor.execute(query, (request_id,))
    req = cursor.fetchone()
    
    if not req:
        cursor.close()
        conn.close()
        return jsonify({'error': 'درخواست یافت نشد'}), 404
    
    if req['status'] == 'pending':
        cursor.execute("UPDATE requests SET status = 'in_progress' WHERE id = %s", (request_id,))
        conn.commit()
        req['status'] = 'in_progress'
    
    cursor.close()
    conn.close()
    
    return jsonify({'request': req})

@app.route('/api/requests/<int:request_id>/response', methods=['POST'])
@token_required
def send_response(admin, request_id):
    data = request.json
    response_text = data.get('response_text')
    
    if not response_text:
        return jsonify({'error': 'پاسخ خالی است'}), 400
    
    conn = get_db()
    cursor = conn.cursor()
    
    cursor.execute(
        """UPDATE requests SET 
           response_text = %s, 
           status = 'completed'
           WHERE id = %s""",
        (response_text, request_id)
    )
    
    conn.commit()
    cursor.close()
    conn.close()
    
    return jsonify({'success': True, 'message': 'پاسخ ارسال شد'})

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=5000, debug=True)

application = app